function main

% -----------
% Analysis of Traveling Waves using Kuramoto model and Connectome data
% 2021.09.10 first vesion (no readme.txt)
%
% This matlab code shows a series of figures used in the paper: 
%
% Procedure:
% step 0: set parameters
% step 1: calc & save time-series data
% step 2: calc traveling waves
% step 3: Figure 2 (traveling waves)
% step 4: Figure 3 (spatial components)
% step 5: Figure 4 (tempora segmentation)
% step 6: Figure 5 (unidirectional communication)

% ======================================================================
% =================================== [step 0] set parameters 

% --------------------- load connectivity data
n=468; % #units
load('./data/flat_maps.mat')   % 2D-maps and MNI location
load('./data/hCP1065_region468_n.mat') ; % connections, ww and w_len [mm]
yeo =load('./data/yeo2011_region468.mat') ; % resting-state connectivitiy

% --------------------- calc. delay matrix
mn_len = mean( w_len(ww(:) > 0 ) ); 
t_del =  w_len /mn_len  * 6; % 6 ;   % [ms], mean delay = 6 ms  
fprintf('mn-len:%4.2f [mm], mn-vel:%4.2f [m/s], mn-del:%4.2f [ms]\n',mn_len, mn_len/10, mean(t_del( ww(:) > 0 )) ) ; 

% -------------------- #connections (%),  (node degree)
nn = mean(sign(ww));  % degree , >0.3 hub  

% ------------------- order of regions  (for visualization)
[~, ord1] = sort( xyz0(1:450,2) ,'descend') ;  %  cortical regions
[~, ord2] = sort( xyz0(451:end, 2) ,'descend') ; % thalamus 
ord = [ord1; (ord2+450)] ;

% ------------------ default parameters 
cw_inp = 0.5;   % ### s.d. of internal freq*10 (default: 0.5)
c_w = 2;        % ### coupling strength (k) * 10 (default: 0.5) 
fnm_base = './work/main_base.mat';   % file name of time-series data
t_int = 0.5; % ### interval of the change of intrernal frequency [s]
t_dur = 100; % ### duration of simulation [s] (default: 100)

fnm_vel = './work/main_base_vel.mat';   % filename of velocity data 
fnm_rad = './work/main_base_radial.mat';  % filename of locally radial waves

% ======================================================================
% ================================== [step 1] calculate time-series data
if ~exist( fnm_base ) 
    disp('calculate time series data .. .')
    % ----------------- one example
   
    inp = randn( n,1 ) *cw_inp;  % internal freq. 
    [~, st, ~] = calc_kuramoto_withlag( ww*c_w, t_del/1000, 1, inp) ; % ## first 1s
    
    stock = [];  % phase 
    inpset = []; % internal freq.
    for i=1:round( t_dur/ t_int ) 
        disp( i ) 
        inp= randn( n,1 ) *cw_inp; % ## internal freq 
        [stock2, st, tt0] = calc_kuramoto_withlag( ww*c_w, t_del/1000  , t_int ,inp, st) ;
        stock = [stock, stock2  ]; 
        inpset = [inpset, repmat( inp, 1, size( stock2, 2 ) ) ]; 
    end
    
    tt = (1:size(stock,2)) * (tt0(2)-tt0(1)); % time points
   
    % --------- re-sampling 
    tt = tt(1:4:end);
    stock = stock(:, 1:4:end) ;
    inpset = inpset(:, 1:4:end) +10;
    
    % ---------- save 
    save( fnm_base, 'stock', 'tt', 'inpset'); 
    disp('done.');
else
    load( fnm_base ) ;
end

% ======================================================================
% ================================== [step 2] calculate  traveling waves
if ~exist( fnm_vel ) 
    tflg = []; % sampling point 
    for dt = (t_int*3/5):t_int:tt(end)
        tflg=[tflg, find( tt>=dt, 1,'first')];
    end
        
    tic
    v1 = calc_vset( stock(:,:), tflg, t_dur, maps(1) ) ; toc;
    v2 = calc_vset( stock(:,:), tflg, t_dur, maps(2) ) ; toc;
    
    save( fnm_vel, 'v1', 'v2', 'tflg') ;
else
    load( fnm_vel ) ;
end


% =====================================================================
% ==================================== [step 3] Figure 2 (traveling waves)
clf
colormap( hsv )

itrial = 3;
tflg1 = tflg(itrial) ;  % a sampled time point
tbin = tt(2) - tt(1) ; % s/point

subplot(6,1,1);  % -----------------time series of  internal frequency
imagesc(tt, 1:n, inpset(ord,:), [-1,1]*1.2 +10) ;
hold on
plot( [1 1] * tflg1*tbin, [1,27],'w-');
for dt = t_int:t_int:tt(end)
    plot( [1 1] * dt, [1,n],'k-');
end
hold off
xlabel('Time (s)');
ylabel('Region ID');
set( gca,'xtick', 0:0.5:2) ;
axis( [0,2, 1,n ] )
colorbar

subplot(6,1,2);  % --------------time series of phase 
imagesc(tt, 1:n, mod(stock(ord,:),2*pi) , [0,2*pi] ) ;
hold on
plot( [1 1] * tflg1*tbin, [1,n],'w-');
for dt=0.5:0.5:tt(end)
    plot( [1 1] * dt, [1,n],'k-');
end
hold off
xlabel('Time (s)');
ylabel('Region ID');
set( gca,'xtick', 0:0.5:2) ;
axis( [0,2, 1,n ] )
colorbar

subplot(3,3, [4,5]);  % ------------ surfae map of phase at a time 
pos = [maps(1).mapx_p, maps(1).mapy_p];
vec1 = [v1.x(:,itrial), v1.y(:,itrial)] .* [v1.vel(:,itrial),  v1.vel(:,itrial)] ;
vec1( find(v1.vel(:,itrial) <0.2 | v1.vel(:,itrial) > 20 ) , :) = NaN;

a1 = [stock( :, tflg1 )  ]  ; %
a1 = mod( a1, 2*pi ) ;
plot_pos_vec2( pos,  vec1 *0.02, a1, maps(1).maplines, [0, 2*pi]);  % a1
hold on
plot( min(maps(1).mapx_p) +  [0,1]* maps(1).del20mm/20*50, [ 1,1] * (-0.9),'k-') ;
hold off
title(sprintf('Time = %3.1f', tt(tflg(itrial)))) ;
axis off

subplot(3,3,6); % ------------------ histogram of speed 
bw1 = 2;  % bin for velocity histogram
bbin = bw1:bw1:20;

vv1 = v1.vel( ~isnan( v1.vel(:) ) ) ;
bb1= hist( vv1( vv1>0.2 & vv1 < max(bbin) ) , bbin ) ;
vv2 = v2.vel( ~isnan( v2.vel(:) ) ) ;
bb2= hist( vv2( vv2>0.2 & vv2 < max(bbin) ) , bbin ) ;

plot(bbin, bb1/sum(bb1)*100,'k.-');
hold on
plot( bbin, bb2/sum(bb2)*100, 'kx:') ;
hold off
xlabel('Speed (m/s)')
ylabel('Frequency (%)')
set( gca, 'box', 'off') ;
fprintf('# surface: median: %4.2f (%4.2f-%4.2f), mean: %4.2f (%4.2f)\n', ...
    median( vv1), quantile(vv1,0.25),quantile(vv1,0.75), mean(vv1), std(vv1) );
fprintf('# on-gyrus: median: %4.2f (%4.2f-%4.2f), mean: %4.2f (%4.2f)\n', ...
    median( vv2), quantile(vv2,0.25),quantile(vv2,0.75), mean(vv2), std(vv2) );
legend( {'Cortex', 'Gyrus'} )

% ----  topographic maps(average)
for imap1 = 1:2
    if imap1==1
        subplot(3,3,[7,8]);
        vv = v1;
    else
        subplot(3,3,9) ;
        vv=v2;
    end
    
    pos = [maps(imap1).mapx_p, maps(imap1).mapy_p];
    xx = vv.x(:,:) .* vv.vel(:,:) ;
    yy = vv.y(:,:) .* vv.vel(:,:) ;
    vec1 = [ mean( xx, 2, 'omitnan') , mean( yy, 2, 'omitnan') ]  ;
    
    plot_pos_vec2( pos,  vec1 *0.02, [], maps(imap1).maplines, [0, 2*pi]);  % a1
    hold on
    plot( min(maps(imap1).mapx_p) +  [0,1]* maps(imap1).del20mm/20*50, [ 1,1] * (-0.9),'k-') ;
    hold off
    axis off
    title('Average') ;
end

input('Hit Enter to show net figure. [RET]');

% =====================================================================
% ==================================[step 4] Figure 3 (spatial components)
clf  
colormap( jet ); 

% -------- set relative phase
tfr = exp( 1i * stock(1:450, : ) ) ;
a1 =  exp( 1i * angle( sum( tfr ) ) );
b = angle( tfr  .*  repmat( conj(a1), size(tfr,1),1) ); % relative phase
fprintf('#Relative phae (overall), std=%3.2f (rad)\n', std( b(:)) ) ;
 
% -------- spatial components
[co1, sc1, la,~,ep1] = pca( b( 1:450, : ) ,'Centered' ,'off');  % # PCA
fprintf('#PC contributions: %3.2f, %3.2f, %3.2f, %3.2f, %3.2f\n', ep1(1:5) /100) ;

subplot(6,1,1); % ------------------------- time series of relative phase 
imagesc(tt, 1:450, b(ord1,:)  ,[-1 1]*pi ) ;
hold on
for dt=0.5:0.5:tt(end)
    plot( [1 1] * dt, [1,450],'k-');
end
hold off
axis( [0,5, 1,450 ] )
xlabel('Time (s)');
ylabel('Region ID');
c = colorbar;
c.Label.String ='Relative phase';

subplot(4,3,[7,8] - 3 ); % ------------------- surface map of PC1
imap1 = 1;
pos = [maps(imap1).mapx_p, maps(imap1).mapy_p];
dum = [sc1(:,1); zeros(468-450,1 )];
plot_pos_vec2( pos,  [] , dum/ max(abs(dum)) * pi, maps(imap1).maplines);  % #
hold on
plot( min(maps(imap1).mapx_p) +  [0,1]* maps(imap1).del20mm/20*50, [ 1,1] * (-0.9),'k-') ;
hold off
axis off
title('PC1') ;
    
subplot(4,3,6) ; % ------------------------- contributions
bar(1:5, ep1(1:5) , 'facecolor', [1 1 1] * 0.8 ) ;
xlabel('PC');
ylabel('Contribution (%)') ;
set( gca, 'box', 'off') ;
title('Relative phase') ;
    
 
subplot(4,3,[7,8]) ;  % ------------------- image of resting-state network 
load( './data/yeo_map.mat' ) ; % load the image of flatmap of yeo2011  
imagesc( [-2,2]+1/100, [-1,1]+1/100, ye_map, [0,7] ) ;
axis equal
ccc = colormap('jet');
ccc(1,:) = 1;
colormap( ccc ) ;
hold on
for k=1:length( maps(imap1).maplines )
    plot( maps(imap1).maplines(k).xy(:,1), maps(imap1).maplines(k).xy(:,2),'k-');
end
hold off
axis xy  equal
set( gca, 'xdir','reverse') ;
axis off
title('Functional Networks') ;
    
% ----- distibution of phase in the empirical network
subplot( 4,3,12 -3 ) ;
b1 = sc1(1:450, 1) / std( sc1(1:450, 1) )  ;
for iy=1:7
    ddd{iy} =b1( yeo.ov(1:450) == iy )   ;    % yeo.ov
    ccc7{iy} =  ccc( round(iy * 36.5) , : );
    mm(iy) = median( ddd{iy} ) ;
end

violin_plot_rev( ddd, [-1,1] * 2.5, ccc7) ;   % #

title('PC1');
set( gca, 'yticklabel', yeo.la, 'ytick', 1:7);
xlabel('PC score (Z)') ;
set(gca,  'ydir','normal')

input('Hit Enter to show net figure. [RET]'); 

% =======================================================================
% ==================================[step 5] Figure 4 (tempora segmentation)
clf

% --- calc average in the empirical networks 
inpfn = []; % internal frequency, mean
for k=1:7
    inpfn = [ inpfn ; mean(inpset( yeo.ov(1:450) == k , : ) ) ];
end
     
pm = [];   % phase, mean
ps = [];   % phase, std
for k=1:7
    pm = [ pm ; mean( b( yeo.ov(1:450) == k , : ) )  ];  % ./ std( b( yeo.ov(1:450) == k , : ) )
    ps = [ ps ; std( b( yeo.ov(1:450) == k , : ) )  ];
end
nfn = hist( yeo.ov(1:450) , 1:7 ) ; % # regions

% --------- effect size d, in time series
dum_d=[];
w_d_av = zeros( 7 );
for k = 1:7
    dumk = [];
    for l=find( [1:7]~=k )
        sc =sqrt( ( nfn(l) *ps(l,:).^2 +nfn(k) *ps(k,:).^2 ) / (nfn(k) + nfn(l) ) );
        d1 = (pm(l,:) - pm(k)) ./ sc;  % effect size 
        dumk = [dumk; d1];
        
        w_d_av(k,l) = mean( abs(d1)) ;
        w_d_av(l,k) = w_d_av(k,l);
    end
    dum_d = [dum_d; mean( abs(dumk) )];
end
     
subplot(5,1,1) ;  % --------- average phase in the network, time series
hold on
for k=7:(-1): 1 % 1:7
    plot( tt, pm(k,:), 'color', ccc( round(k * 36.5) , : ) ) ;
end
for dt = 0.5:0.5:max(tt)
    plot( [1,1] * dt, [-1,1]*0.5, 'k-');
end
hold off
axis( [ 0, 5, -0.49,0.49] )
xlabel('Time (s)') ;
ylabel('Relative phase (mean)');
legend( yeo.la(end:(-1):1) )

subplot(3,3,4);  % --- internal freq (av)- rel.phase(av) relationshp
hold on
for k=1:7
    plot( inpfn(k,tflg), pm(k,tflg) ,'.', 'color',  ccc( round(k * 36.5) , : )  ) ; % tflg
end
hold off
xlabel('Internal frequency (av)');
ylabel('Relative phase (av)');
axis( [ -0.22+10, 0.22+10, -0.5, 0.5] ) ;

subplot(3,3,5) ;  % ----- degree - rel.phase (av) relationship 
dum=[];
bmn =[];
hold on
for k=1:7
    dum( k ) = mean( nn(yeo.ov(1:450) == k ) ) ;  % node degree 
    bmn(k) = mean( b( yeo.ov(1:450) == k ,tflg ),'all' ); % rel.phase
end
for k=1:7
    plot( dum(k), bmn(k), 'ko','color', ccc( round(k * 36.5) , : ) ) ;
end
hold off
hold on
p=polyfit( dum ,  bmn,1);
x = [ min( dum) , max( dum) ];
plot( x, polyval(p, x),'k-') ;
hold off
axis( [ 0.1 0.22 -0.3 0.3] );
xlabel('Node degree (av) (%)');
ylabel('Relative phase (av)');
c1 = corr( dum', bmn' ) ;
title(sprintf('corr coef: r=%3.2f', c1) ) ;

subplot(3,3,6);  % ---------------------- matrix of d
imagesc( w_d_av( 7:(-1):1, 7:(-1):1 ) , [0,0.5])
title('d (mean in time)');
colorbar
axis equal
pbaspect( [1 1 1 ] ) ;
set( gca,'xtick',1:7, 'ytick',1:7, 'xticklabel', yeo.la(7:(-1):1), 'yticklabel', yeo.la(7:(-1):1) );

input('Hit Enter to show net figure. [RET]'); 

% =======================================================================
% ==============================[step 6] Figure 5 (unidirectional communication)
clf

dt1 = 2.8; % a sampled time point: one example 
it_sample1=find( tt >=dt1, 1,'first') ;

% --- uci at time dt1 
t_delp = t_del(1:450, 1:450) / 100 *(2*pi)  ; % delay in pahse (10 Hz)
ph2= repmat( b( 1:450, it_sample1 ) , 1, 450 );
ph2b = ph2' - t_delp ;
x = mod(ph2 -ph2b + pi , 2*pi) - pi;
uci1 = (cos( x ) + 1 ) /2 ;  % ## unidirectional communication index 
uci1( t_delp ==0 ) = NaN ;

% ---------   UCI, temporal distribiution
uci_slant = [];
for it = tflg
    inpx = inpset( 1:450 , it );
    ph2= repmat(  b( 1:450, it ) , 1, 450 );
    ph2b = ph2' - t_delp ;
    x = mod(ph2 -ph2b + pi , 2*pi) - pi;
    uci = (cos( x ) + 1 ) /2 ;  % ##
    uci( t_delp ==0 ) = NaN ;
    
    p=polyfit( inpx , mean( uci, 'omitnan')' ,1);
    uci_slant=[uci_slant, p(1)];
end
   
subplot(4,1,1) ;  % ------------- relative phase ordered by internal freq.

bsort = [];  % sorted relative phase 
inp_dum = inpset(1:450,1);
tchange =[1];
for it=1:size( inpset,2 )
    if sum(abs(inpset(1:450,it) - inp_dum))>10^-3
        tchange=[tchange, it];
        inp_dum = inpset(1:450,it);
    end
end
for l=1:(length(tchange)-1)
    it1 = tchange(l);
    it2 = tchange(l+1) ;
    [~,ind] = sort( inpset( 1:450, round( (it1+it2)/2 ) ) ) ;
    bsort = [ bsort, b( ind, it1:it2) ];
end

imagesc(tt(tt<5), 1:450, bsort(:, tt<5), [-1,1]*3 );
hold on
for dt = t_int:t_int:max(tt)
    plot( [1,1] * dt, [1,450], 'k-');
end
hold off
axis( [0,5,1,450]);
xlabel('Time (s)') ;
ylabel('Region ID (sorted by int.freq.)')
c= colorbar;
c.Label.String =  'Relative phase';

subplot(3,3,4); % ---------------internal freq. - rel. pahse, at a time

plot( inpset(1:450, it_sample1), b( 1:450,  it_sample1 ) , 'b.') ;
xlabel('Input');
ylabel('Relative phase') ;
title(sprintf('Time = %3.1f', dt1));
set( gca,'box','off')

subplot(3,3,5); % --------------------- uci at a time
inpx = inpset( 1:450 , it_sample1 );
plot( inpx, mean( uci1, 'omitnan') ,'b.')
hold on
p=polyfit( inpx , mean( uci1, 'omitnan')' ,1);
x = [ min( inpx) , max( inpx )] ;
plot( x, polyval(p, x),'k-') ;
text( -1,0.5, sprintf('Y=%4.2fX+%4.2f',p) ) ;
hold off
xlabel('Internal frequency');
ylabel('UCI') ; 
title(sprintf('Time = %3.1f', dt1));
set( gca,'box','off')

subplot(3,3,6) ;  % -------------------- histgram of uci slant 
hbin = 0:0.01:0.08; 
bh = hist( uci_slant, hbin ) ;
bar( hbin, bh/sum(bh), 1, 'facecolor', [1 1 1] * 0.8 );
xlabel('Slop of efficiency');
ylabel('Frequency');
set( gca, 'box', 'off') ;

input('Hit Enter to show net figure. [RET]'); 

% =======================================================================
% ================================[step 7] Figure 6 (locally radial waves)
clf
    
if  ~exist( fnm_rad )
    d_rad = 20;
    radset = 0:d_rad:80 ;%  ###  set of radius 
     
    imap1=1;
    pos = [maps(imap1).mapx_p, maps(imap1).mapy_p];
    posd = pos/maps(imap1).del20mm * 20; % [mm]
    for k=1:450
        for l=1:450
            vec1 = posd(l,:) - posd(k,:);
            d1  = sqrt( sum( vec1.^2 ) ) ;
            vec1 = vec1 / d1 ;
            dnei(k,l).d1  =  d1;  % --distance
            dnei(k,l).vec1 = vec1; % ---vector from k to l
            dnei(k,l).a  =  atan2( vec1(2), vec1(1) ) ; % angle
        end
    end  
  
    for ir1 = 1:length( radset )
        rad1 = radset( ir1 ) ;
        dum_nei =zeros( 450 * length(tflg) , 3);
        icount = 1;
        for it1 = 1:length(tflg)
            for k=1:450
                lflg = find( sign( pos( k,1 ) ) ==   sign( pos( 1:450,1 ) ) ) ;
                dum_nei_p= [];
                for l=lflg(:)' % 1:450  % only same hemishere
                    d1 = dnei(k,l).d1 ;
                    if d1 <= rad1+d_rad & d1 > rad1
                        vec1 = dnei(k,l).vec1;
                        v1_vec = [v1.x(l,it1),v1.y(l,it1)] .* v1.pg(l,it1) /100; % [rad/cm] 
                        v1_vec_abs = sum(v1_vec.*vec1);
                        dum_nei_p = [dum_nei_p, v1_vec_abs];
                    end
                end
                dum_nei(icount,:) = [ inpset(k, tflg(it1)),  mean(dum_nei_p,'omitnan') , k]; % 
                icount = icount +1 ;
            end
        end
        
        flg =find( ~isnan( dum_nei(:,2)  ) ) ;
       %  plot( dum_nei(flg,1 ) , dum_nei(flg,2),'.') ;
        c1 = corr( dum_nei(flg,1 ) , dum_nei(flg,2) );
        b2 = regress( dum_nei(flg,1 ) , [dum_nei(flg,2) , ones(length(flg), 1 ) ]);
        disp( [c1, b2(1)] );
        
        stockz(ir1).dum_nei = dum_nei;
        % ## dum_nei:  1:inp, 2: index, 3: region ID
    end
    save( fnm_rad, 'stockz', 'radset') ;
else
    load( fnm_rad) ;
end
 
subplot(3,3,1);   % -------------ipnut-phase.gradient relatiohisp

iunit = 17 ;  % one example  
for ir1=1:2
    dum_nei = stockz(ir1).dum_nei;
    flg =find( ~isnan( dum_nei(:,2)  ) & dum_nei(:,3) == iunit) ;
    plot( dum_nei(flg,1 ) , dum_nei(flg,2),'.') ;
    
    hold on
    p=polyfit( dum_nei(flg,1 ) ,  dum_nei(flg,2) ,1);
    x = [ min( dum_nei(flg,1 )) , max( dum_nei(flg,1 ) )] ;
    plot( x, polyval(p, x),'k-' ,'color', [0.6, 0,0] ) ;
    text( -1,0.08*ir1, sprintf('Y=%4.2fX+%4.2f',p) ) ;
end
hold off 

xlabel('Internal frequency') ;
ylabel('Source-sink index') ;
title( sprintf('Two radius, region %d', iunit )) ;
set( gca,'box', 'off') ;

subplot(3,3,2); % ---------------- radius-dependence of SSI slope
ssi_slant = [];
for iunit = 1:450
    dum1 =zeros( 1, length(radset))  * NaN ;
    for ir1 = 1:length( radset )
        dum_nei = stockz(ir1).dum_nei;
        flg =find( ~isnan( dum_nei(:,2)  ) & dum_nei(:,3) == iunit) ;
        
        b2 = regress( dum_nei(flg,2) , [dum_nei(flg,1) , ones(length(flg), 1 ) ]);
        dum1(ir1)=  b2(1);
    end
    ssi_slant = [ssi_slant; dum1 ]  ;
end

errorbar( radset + 15/2,  mean(ssi_slant,'omitnan' ) , std(ssi_slant,'omitnan' )   ,'ko-', 'color', [0.6, 0,0] )
xlabel('Radius (mm)');
ylabel('Slope of SSI');
set( gca,'box', 'off') ;
axis( [0, 100, -0.007, 0.08] ) ;

subplot(3,3,3) ; % ----------------node degree -  SSI slope relatioship
flg =  find( ~isnan(ssi_slant(:,1) )) ;
plot( nn(flg), ssi_slant(flg,1) ,'b.' )
xlabel('Degree') ;
ylabel('Slope of phase gradient (0-20 mm)') ;
set( gca,'box','off')
hold on
p=polyfit( nn(flg), ssi_slant(flg,1)' ,1);
x = [ min( nn(flg)) , max( nn(flg) )] ;
plot( x, polyval(p, x),'k-') ;
hold off
c1 = corr( nn(flg)', ssi_slant(flg,1) ) ;
title(sprintf('corr coef=%4.2f', c1 ) );
axis( [0, 0.42, 0, 0.18] ) ;
return
   

